<?php

namespace App\Helpers;

use HTMLPurifier;
use HTMLPurifier_Config;

class HtmlSanitizer
{
    protected static $purifier;

    /**
     * Sanitize HTML content to prevent XSS attacks
     *
     * @param string $html
     * @return string
     */
    public static function sanitize(?string $html): string
    {
        if (empty($html)) {
            return '';
        }

        if (!self::$purifier) {
            $config = HTMLPurifier_Config::createDefault();
            
            // Set cache directory
            $config->set('Cache.SerializerPath', storage_path('app/purifier'));
            
            // Allow safe HTML tags
            $config->set('HTML.Allowed', 'p,br,strong,em,u,a[href],ul,ol,li,blockquote,code,pre,h1,h2,h3,h4,h5,h6,img[src|alt|width|height]');
            
            // Ensure links open safely
            $config->set('HTML.TargetBlank', true);
            $config->set('HTML.Nofollow', true);
            
            // Allow safe URLs only
            $config->set('URI.AllowedSchemes', ['http' => true, 'https' => true]);
            
            self::$purifier = new HTMLPurifier($config);
        }

        return self::$purifier->purify($html);
    }

    /**
     * Sanitize and strip all HTML tags
     *
     * @param string $html
     * @return string
     */
    public static function stripAll(?string $html): string
    {
        return strip_tags($html ?? '');
    }
}